/* Copyright 2007 by Kim Minh Kaplan
 *
 * greyfix.c version 0
 *
 * Postfix policy daemon designed to prevent spam using the
 * greylisting method.
 *
 * Greylisting: http://projects.puremagic.com/greylisting/
 * Postfix: http://www.postfix.org/
 * Kim Minh Kaplan: http://www.kim-minh.com/
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <sys_defs.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <unistd.h>
#include <syslog.h>
#include <sys/stat.h>
/* global library */
#include <mymalloc.h>
#include <stringops.h>
#include <msg_syslog.h>
#include <msg.h>
#include <mail_task.h>
#include <mail_conf.h>
#include <mail_params.h>
#include <mail_dict.h>

#include <db.h>

/**
 * This determines how many seconds we will block inbound mail that is
 * from a previously unknown (ip, from, to) triplet.  If it is set to
 * zero, incoming mail association will be learned, but no deliveries
 * will be tempfailed.  Use a setting of zero with caution, as it will
 * learn spammers as well as legitimate senders.
 **/
#define DELAY_MAIL_SECS (58 * 60)	/* 58 minutes */
/**
 * This determines how many seconds of life are given to a record that
 * is created from a new mail [ip,from,to] triplet.  Note that the
 * window created by this setting for passing mails is reduced by the
 * amount set for $delay_mail_secs.  NOTE: See Also:
 * update_record_life and update_record_life_secs.
 */
#define AUTO_RECORD_LIFE_SECS (5 * 3600) /* 5 hours */

#define CONF_DB_HOME PACKAGE"_directory"
#define DEF_DB_HOME DATA_STATE_DIR"/"PACKAGE

#define DB_FILE_NAME "triplets"
#define SEP '\000'

#define prefixp(s,p) (!strncmp((s),(p),sizeof(p)-1))

struct triplet_data {
    time_t create_time;
    time_t access_time;
    unsigned long block_count;
    unsigned long pass_count;
};

static const char str_smtp[] = "SMTP";
static const char str_esmtp[] = "ESMTP";
static const char str_rcpt[] = "RCPT";
static const char str_action[] = "action=";

static DB_ENV *dbenv = 0;
static DB *db = 0;

static char *policy_request = 0;
static size_t policy_request_size = 0;
static size_t policy_request_fill = 0;

static DBT dbkey = { 0 };
static DBT dbdata = { 0 };
static struct triplet_data triplet_data;

static int debug_me = 0;

/**********************************************************************
 * Berkeley DB routines
 */
static void
log_db_error(const char *msg, int error)
{
    msg_error("%s: %s", msg, db_strerror(error));
}

static void
db_errcall_fcn(const DB_ENV *dbenv, const char *errpfx, const char *msg)
{
    msg_error("%s: %s", errpfx ? errpfx : "Berkeley DB", msg);
}

static int
prepare_env()
{
    int rc;
    const char *home = mail_conf_eval(get_mail_conf_str(CONF_DB_HOME, DEF_DB_HOME, 0, 0));
    rc = db_env_create(&dbenv, 0);
    if (rc)
	log_db_error("db_env_create", rc);
    else {
	dbenv->set_errcall(dbenv, db_errcall_fcn);
	rc = dbenv->open(dbenv, home,
			 DB_INIT_CDB | DB_INIT_MPOOL | DB_CREATE, 0);
	if (rc)
	    log_db_error("dbenv->open", rc);
    }
    return rc;
}

static int
prepare_db()
{
    int rc;
    rc = db_create(&db, dbenv, 0);
    if (rc)
	log_db_error("db_create", rc);
    else {
	rc = db->open(db, NULL, DB_FILE_NAME, NULL, DB_BTREE,
		      DB_CREATE, 0644);
	if (rc)
	    log_db_error("db->open", rc);
    }
    return rc;
}

static int
initialize()
{
    int rc;
    char *version;
    int major, minor, patch;
    version = db_version(&major, &minor, &patch);
    if (DB_VERSION_MAJOR != major || DB_VERSION_MINOR != minor) {
	msg_fatal("This daemon was compiled with " DB_VERSION_STRING " (%d.%d.%d) definitions "
		  "but it is linked with %s (%d.%d.%d).  This will not work!  "
		  "Check that the version of the developpement files for Berkeley DB "
		  "match the version that was linked in your Postfix program.",
		  DB_VERSION_MAJOR, DB_VERSION_MINOR, DB_VERSION_PATCH,
		  version, major, minor, patch);
	abort();
    }
    if (DB_VERSION_PATCH != patch && (msg_verbose || debug_me))
	msg_info("Compiled with " DB_VERSION_STRING " (%d.%d.%d) definitions.  "
		 "Running with %s (%d.%d.%d).",
		 DB_VERSION_MAJOR, DB_VERSION_MINOR, DB_VERSION_PATCH,
		 version, major, minor, patch);
    else if (debug_me)
	msg_info("This daemon was compiled with " DB_VERSION_STRING " (%d.%d.%d) definitions.",
		 DB_VERSION_MAJOR, DB_VERSION_MINOR, DB_VERSION_PATCH);
    dbdata.data = &triplet_data;
    dbdata.size = sizeof triplet_data;
    dbdata.ulen = sizeof triplet_data;
    dbdata.flags = DB_DBT_USERMEM;
    rc = prepare_env();
    if (!rc)
	rc = prepare_db();
    return rc;
}

static void
cleanup()
{
    int rc;
    if (dbkey.data)
	free(dbkey.data);
    if (policy_request)
	free(policy_request);
    if (db) {
	rc = db->close(db, 0);
	db = 0;
	if (rc)
	    log_db_error("DB close", rc);
    }
    if (dbenv) {
	rc = dbenv->close(dbenv, 0);
	dbenv = 0;
	if (rc)
	    log_db_error("DB_ENV close", rc);
    }
    if (debug_me)
	msg_info("Cleaned");
}

static void
fatal(const char *msg)
{
    int err = errno;;
    cleanup();
    msg_fatal("%s: %s", msg, db_strerror(err));
    abort();
}

static void *
xrealloc(void *ptr, size_t size)
{
    void *newptr = realloc(ptr, size);
    if (newptr)
	return newptr;
    fatal("Out of memory in xrealloc");
}

static void
build_triplet_key(const char *ip, const char *from, const char *to)
{
    const char *endip = strchr(ip, '\n'),
	*endfrom = strchr(from, '\n'),
	*endto = strchr(to, '\n');
    size_t lenip = endip - ip,
	lenfrom = endfrom - from,
	lento = endto - to;
    size_t total = lenip + lenfrom + lento + 2;
    char *buf;
    if (dbkey.ulen < total) {
	dbkey.data = xrealloc(dbkey.data, total);
	dbkey.ulen = total;
	dbkey.flags = DB_DBT_USERMEM;
    }
    dbkey.size = total;
    buf = (char*)dbkey.data;
    memcpy(buf, ip, lenip);
    buf += lenip;
    *buf++ = 0;
    memcpy(buf, from, lenfrom);
    buf += lenfrom;
    *buf++ = 0;
    memcpy(buf, to, lento);
}

static void
touch_data()
{
    if (time(&triplet_data.access_time) == (time_t)-1)
	fatal("time failed");
}

static void
build_data()
{
    touch_data();
    triplet_data.create_time = triplet_data.access_time;
    triplet_data.block_count = 0;
    triplet_data.pass_count = 0;
}

/**********************************************************************
 * See SMTPD_POLICY_README
 */

static void
safe_writes(int fd, const char *s)
{
    size_t len = strlen(s);
    while (len) {
	int n = write(fd, s, len);
	if (n > 0)
	    len -= n;
	else if (n == 0)
	    msg_error("Retrying due to empty write");
	else if (errno == EINTR && debug_me)
	    msg_info("Retrying write: %s", strerror(errno));
    }
}

/* XXX Assumes there is an empty line marking the end of request */
static char *
find_attribute(const char *name)
{
    char *p;
    size_t nlen;
    nlen = strlen(name);
    for (p = policy_request; *p != '\n'; p = strchr(p, '\n') + 1)
	if (strncmp(name, p, nlen) == 0 && p[nlen] == '=') {
/*
	    if (debug_me) {
		char *e = strchr(p, '\n');
		*e = 0;
		msg_info("Attribute %s", p);
		*e = '\n';
	    }
*/
	    return p + nlen + 1;
	}
    return NULL;
}

static char *
find_empty_line(const char *p, const char *const endp)
{
    while (p < endp && (p = memchr(p, '\n', endp - p))) {
	p++;
	if (p < endp && *p == '\n')
	    return (char *) p + 1;
    }
    return NULL;
}

/* Find the end of request marker */
static char *
find_eor()
{
    return find_empty_line(policy_request,
			   policy_request + policy_request_fill);
}

/* Forget the oldest policy request */
static void
forget_policy_request()
{
    const char *eor = find_eor();
    if (eor) {
	policy_request_fill -= eor - policy_request;
	memmove(policy_request, eor, policy_request_fill);
    }
    else
	policy_request_fill = 0;
}

/* Read in a new SMTPD access policy request */
static const char *
read_policy_request(int in)
{
    forget_policy_request();
    while (! find_eor()) {
	size_t wanted;
	int n;
	/* Make sure there is some room to read data */
	if (policy_request_fill == policy_request_size) {
	    if (policy_request_size)
		policy_request_size *= 2;
	    else
		policy_request_size = BUFSIZ;
	    if (debug_me)
		msg_info("allocate %u bytes for request buffer",
			 policy_request_size);
	    policy_request = xrealloc(policy_request, policy_request_size);
	}
	wanted = policy_request_size - policy_request_fill;
	n = read(in, policy_request + policy_request_fill, wanted);
	if (n < 0)
	    log_db_error("read_policy_request failed", n);
	else if (n)
	    policy_request_fill += n;
	else
	    return NULL;
    }
    return policy_request;
}

static void
get_grey_data()
{
    int rc;
    rc = db->get(db, NULL, &dbkey, &dbdata, 0);
    if (rc == DB_NOTFOUND)
	build_data();
    else if (rc) {
	log_db_error("get failed", rc);
	fatal("Exiting");
    }
    else
	touch_data();
}

static void
put_grey_data()
{
    int rc;
    rc = db->put(db, NULL, &dbkey, &dbdata, 0);
    if (rc)
	log_db_error("put", rc);
}

static const char *
process_smtp_rcpt()
{
    const char *action = "DUNNO";
    get_grey_data();
    /* Expire record created from a new mail (ip, from, to) */
    if (triplet_data.pass_count == 0
	&& triplet_data.access_time - triplet_data.create_time > AUTO_RECORD_LIFE_SECS) {
	if (debug_me)
	    msg_info("expired record");
	triplet_data.create_time = triplet_data.access_time;
    }
    /* Block inbound mail that is from a previously unknown (ip, from, to) triplet */
    if (triplet_data.access_time - triplet_data.create_time < DELAY_MAIL_SECS) {
	triplet_data.block_count++;
	action = "DEFER_IF_PERMIT Greylisted, try again latter.  See http://projects.puremagic.com/greylisting/ for more information.";
    }
    else
	triplet_data.pass_count++;
    put_grey_data();
    return action;
}

int
main(int argc, char **argv)
{
    int rc;
    if (getenv(CONF_ENV_VERB))
        msg_verbose = 1;
    if (getenv(CONF_ENV_DEBUG))
        debug_me = 1;
    var_procname = mystrdup(basename(argv[0]));
    set_mail_conf_str(VAR_PROCNAME, var_procname);
    msg_syslog_init(mail_task(var_procname), LOG_PID, LOG_FACILITY);
    if (msg_verbose)
        msg_info("daemon started");
    mail_conf_read();
    mail_dict_init();
    rc = initialize();
    if (rc) {
	errno = rc;
	fatal("initialization failure");
    }
    while (read_policy_request(0)) {
	const char *protocol = 0, *state = 0, *ip, *from, *to;
	const char *action = "DUNNO";
	if ((protocol = find_attribute("protocol_name"))
	    && (state = find_attribute("protocol_state"))
	    && (prefixp(protocol, str_smtp) || prefixp(protocol, str_esmtp))
	    && prefixp(state, str_rcpt)
	    && (ip = find_attribute("client_address"))
	    && (from = find_attribute("sender"))
	    && (to = find_attribute("recipient"))) {
	    build_triplet_key(ip, from, to);
	    action = process_smtp_rcpt();
	}
	else if (debug_me) {
	    char *p = 0, *s = 0;
	    if (protocol) {
		p = strchr(protocol, '\n');
		*p = 0;
	    }
	    if (state) {
		s = strchr(state, '\n');
		*s = 0;
	    }
	    msg_info("Ignoring protocol state %s",
		     protocol ? protocol : "(not defined)",
		     state ? state : "(not defined)");
	    if (p)
		*p = '\n';
	    if (s)
		*s = '\n';
	}
	safe_writes(1, "action=");
	safe_writes(1, action);
	safe_writes(1, "\n\n");
    }
    cleanup();
    if (msg_verbose)
	msg_info("daemon stopped");
    return 0;
}
