/*   This is RIFFL, the Replacement iffparse.library, a collection of
 *   routines to read IFF files.
 *
 *   Copyright (C) 1997 Kim-Minh Kaplan.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License version 2 as published by the Free Software Foundation.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file named COPYING for more
 *   details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* TODO: WriteChunkRecords, PropChunks, StopChunks, CollectionChunk,
   CollectionChunks, CloseClipboard, FindCollection,
   SetLocalItemPurge, InitIFFasDOS, InitIFFasClip.  */

#define abs abs			/* shutdown Amiga's redefinition */

#include <utility/hooks.h>
#include <libraries/iffparse.h>
#include <proto/alib.h>
#include <proto/exec.h>
#include <proto/iffparse.h>
#include <proto/utility.h>

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef BUFSIZ
#define BUFSIZ			1024
#endif

#define MAKE_ID_FROM_BUFF(p)	MAKE_ID((p)[0],(p)[1],(p)[2],(p)[3])
#define MAKE_ULONG_FROM_BUFF(p)	MAKE_ID_FROM_BUFF(p)
#define CHAR_FROM_ID(i,n)	((char)(0xff&(((unsigned long)i)>>(8*(3-(n))))))
#define COMPOSITE_P(i)	((i)==ID_FORM||(i)==ID_LIST||(i)==ID_CAT||(i)==ID_PROP)

#define ENTRY(h)	((HookEntry)((h)->h_SubEntry == NULL \
				     ? (h)->h_Entry	     \
				     : (h)->h_SubEntry))

#define MIN(a,b)	((a) < (b) ? (a) : (b))

typedef unsigned long (*HookEntry)(struct Hook *, void *, void *);

/* Private fields of struct LocalContextItem */
struct _LocalContextItem {
    struct LocalContextItem lci_Pub;
};

/* Private fields of struct ContextNode */
struct _ContextNode {
    struct ContextNode	cn_Pub;
    struct MinList	cn_LCIs;
    int			cn_IsBufferred;
    struct MinList	cn_Buffers;
};

/* _ContextNode->cn_Buffers is a list of ContextBuffer */
struct ContextBuffer {
    struct MinNode	cb_Node;
    long		cb_Size;
    /* and after comes cb_Size bytes */
};

/* Private fields of struct IFFHandle */
struct _IFFHandle {
    struct IFFHandle	iff_Pub;
    /* Infos regarding the stream */
    struct Hook	       *iff_StreamHook;
    struct IFFStreamCmd	iff_StreamAction;
    long		iff_RWMode;
    /* Contextual information */
    struct MinList	iff_Contexts;
    struct MinList	 iff_RootLCIs;
    /* Parser information */
    enum { IFFPS_INITIAL, IFFPS_ENTER, IFFPS_EXIT } iff_ParseState;
};

/* Chunk handlers structures */
struct ChunkHandler {
    struct Hook	       *ch_Hook;
    unsigned char      *ch_Object;
};

/* IFF ID & Types functions */
char *
IDtoStr(long id, char *buf)
{
    int i;
    for (i = 0; i < 4; i++)
	buf[i] = CHAR_FROM_ID(id, i);
    buf[i] = 0;
    return buf;
}

int
GoodID(unsigned long id)
{
    int space;
    int i;
    return 1;
    for (i = 0, space = 0; i < 4; i++) {
	char c = CHAR_FROM_ID(id, i);
	if (c < ' ' || c > '~' || iscntrl(c) || (space && isprint(c)))
	    return 0;
	if (c == ' ')
	    space = 1;
    }
    return 1;
}

int
GoodType(unsigned long type)
{
    int i;
    if (! GoodID(type))
	return 0;
    for (i = 0; i < 4; i++) {
	char c = CHAR_FROM_ID(type, i);
	if (islower(c) || ispunct(c))
	    return 0;
    }
    return 1;
}

/* LocalContextItems functions */
struct LocalContextItem *
AllocLocalItem(unsigned long type, unsigned long id, unsigned long ident,
	       unsigned long usize)
{
    struct LocalContextItem *lci =
	malloc(sizeof(struct _LocalContextItem) + usize);
    if (lci != NULL) {
	lci->lci_ID = id;
	lci->lci_Type = type;
	lci->lci_Ident = ident;
    }
    return lci;
}

void
FreeLocalItem(struct LocalContextItem *lci)
{
    free(lci);
}

static void
LocalItemPurge(struct LocalContextItem *lci)
{
    FreeLocalItem(lci);
}

unsigned char *
LocalItemData(struct LocalContextItem *lci)
{
    return (unsigned char *)(((struct _LocalContextItem *)lci) + 1);
}

static struct LocalContextItem *
FindLocalItemInList(struct MinList *list, unsigned long type,
		    unsigned long id, unsigned long ident)
{
    struct LocalContextItem *lci;
    for (lci = (struct LocalContextItem *)list->mlh_Head;
	 lci->lci_Node.mln_Succ != NULL;
	 lci = (struct LocalContextItem *)lci->lci_Node.mln_Succ)
	if (lci->lci_ID == id
	    && lci->lci_Type == type
	    && lci->lci_Ident == ident)
	    return lci;
    return NULL;
}

struct LocalContextItem *
FindLocalItem(struct IFFHandle *iff, unsigned long type, unsigned long id,
	      unsigned long ident)
{
    struct LocalContextItem *lci = NULL;
    struct ContextNode *cn;
    for (cn = CurrentChunk(iff);
	 cn != NULL && lci == NULL;
	 cn = ParentChunk(cn))
	lci = FindLocalItemInList
	    (&((struct _ContextNode *)cn)->cn_LCIs, type, id, ident);
    if (lci == NULL)
	lci = FindLocalItemInList(&((struct _IFFHandle *)iff)->iff_RootLCIs,
				  type, id, ident);
    return lci;
}

static void
StoreItemInList(struct MinList *list, struct LocalContextItem *lci)
{
    /* First look if there is already a similar LCI */
    struct LocalContextItem *p =
	FindLocalItemInList(list, lci->lci_Type, lci->lci_ID, lci->lci_Ident);
    if (p != NULL) {
	Remove((struct Node *)&p->lci_Node);
	LocalItemPurge(p);
    }
    AddHead((struct List *)list, (struct Node *)&lci->lci_Node);
}

/* ContextNode functions */
static struct _ContextNode *
AllocContextNode(void)
{
    return malloc(sizeof(struct _ContextNode));
}

static void
InitContextNode(struct _ContextNode *_cn)
{
    _cn->cn_Pub.cn_Scan = 0;
    _cn->cn_IsBufferred = 0;
    NewList((struct List *)&_cn->cn_LCIs);
    NewList((struct List *)&_cn->cn_Buffers);
}

static void
FreeContextNode(struct _ContextNode *_cn)
{
    struct LocalContextItem *lci;
    struct Node *n;
    while ((lci = (struct LocalContextItem *)
	    RemTail((struct List *)&_cn->cn_LCIs)) != NULL)
	LocalItemPurge(lci);
    while ((n = RemTail((struct List *)&_cn->cn_Buffers)) != NULL)
	free(n);
    free(_cn);
}

/* Context stack */
static struct _ContextNode *
PushContext(struct _IFFHandle *_iff, unsigned long id, unsigned long type,
	    unsigned long size)
{
    struct _ContextNode *_cn = AllocContextNode();
    if (_cn != NULL) {
	InitContextNode(_cn);
	_cn->cn_Pub.cn_ID = id;
	_cn->cn_Pub.cn_Type = type;
	_cn->cn_Pub.cn_Size = size;
	_iff->iff_Pub.iff_Depth++;
	AddTail((struct List *)&_iff->iff_Contexts,
		(struct Node *)&_cn->cn_Pub.cn_Node);
    }
    return _cn;
}

static struct _ContextNode *
_PopContext(struct _IFFHandle *_iff)
{
    _iff->iff_Pub.iff_Depth--;
    return (struct _ContextNode *)RemTail((struct List *)&_iff->iff_Contexts);
}

static void
PopContext(struct _IFFHandle *_iff)
{
    FreeContextNode(_PopContext(_iff));
}

struct ContextNode *
CurrentChunk(struct IFFHandle *iff)
{
    if (iff->iff_Depth > 0)
	return (struct ContextNode *)
	    ((struct _IFFHandle *)iff)->iff_Contexts.mlh_TailPred;
    else
	return NULL;
}

struct ContextNode *
FindPropContext(struct IFFHandle *iff)
{
    struct ContextNode *cn = ParentChunk(CurrentChunk(iff));
    while (cn != NULL && cn->cn_ID != ID_FORM && cn->cn_ID != ID_LIST)
	cn = ParentChunk(cn);
    return cn;
}

/* XXX -- WARNING: the cn_Scan field is invalid.  It is only up to
   date for the CurrentChunk ! */
struct ContextNode *
ParentChunk(struct ContextNode *cn)
{
    struct ContextNode *pn = (struct ContextNode *)cn->cn_Node.mln_Pred;
    return pn->cn_Node.mln_Pred == NULL ? NULL : pn;
}

/* Local Context Items & Contexts */
void
StoreItemInContext(struct IFFHandle *iff, struct LocalContextItem *lci,
		   struct ContextNode *cn)
{
    StoreItemInList(&((struct _ContextNode *)cn)->cn_LCIs, lci);
}

int
StoreLocalItem(struct IFFHandle *iff, struct LocalContextItem *lci,
	       int position)
{
    int error = 0;
    struct ContextNode *cn;
    switch (position) {
    case IFFSLI_ROOT:
	StoreItemInList(&((struct _IFFHandle *)iff)->iff_RootLCIs, lci);
	break;
    case IFFSLI_TOP:
	if ((cn = CurrentChunk(iff)))
	    StoreItemInContext(iff, lci, cn);
	else
	    StoreItemInList(&((struct _IFFHandle *)iff)->iff_RootLCIs, lci);
	break;
    case IFFSLI_PROP:
	if ((cn = FindPropContext(iff)))
	    StoreItemInContext(iff, lci, cn);
	else
	    error = IFFERR_NOSCOPE;
	break;
    }
    return error;
}

/* Chunk handlers */
static int
Handler(struct IFFHandle *iff, unsigned long type, unsigned long id,
	int position, struct Hook *hook, void *object,
	unsigned long ident)
{
    struct LocalContextItem *lci =
	AllocLocalItem(type, id, ident, sizeof(struct ChunkHandler));
    struct ChunkHandler *ch;
    if (lci == NULL)
	return IFFERR_NOMEM;
    ch = (struct ChunkHandler *)LocalItemData(lci);
    ch->ch_Hook = hook;
    ch->ch_Object = object;
    return StoreLocalItem(iff, lci, position);
}

int
EntryHandler(struct IFFHandle *iff, unsigned long type, unsigned long id,
	     int position, struct Hook *hook, void *object)
{
    return Handler(iff, type, id, position, hook, object, IFFLCI_ENTRYHANDLER);
}

int
ExitHandler(struct IFFHandle *iff, unsigned long type, unsigned long id,
	    int position, struct Hook *hook, void *object)
{
    return Handler(iff, type, id, position, hook, object, IFFLCI_EXITHANDLER);
}

static long
CallHandler(struct _IFFHandle *_iff, unsigned long ident, int cmd)
{
    struct ContextNode *cn = CurrentChunk(&_iff->iff_Pub);
    struct LocalContextItem *lci =
	FindLocalItem(&_iff->iff_Pub, cn->cn_Type, cn->cn_ID, ident);
    if (lci != NULL) {
	struct ChunkHandler *ch = (struct ChunkHandler *)LocalItemData(lci);
	return (long)CallHookPkt(ch->ch_Hook, ch->ch_Object, (void *)cmd);
    } else
	return 0;
}

static long
CallEntryHandlers(struct _IFFHandle *_iff)
{
    return CallHandler(_iff, IFFLCI_ENTRYHANDLER, IFFCMD_ENTRY);
}

static long
CallExitHandlers(struct _IFFHandle *_iff)
{
    return CallHandler(_iff, IFFLCI_EXITHANDLER, IFFCMD_EXIT);
}

/* Stop chunk handlers */
static long
StopChunkHandler(struct Hook *hook, void *object, void *cmd)
{
    return IFF_RETURN2CLIENT;
}
static struct Hook StopChunkHook = {{NULL}, 0, StopChunkHandler};

int
StopChunk(struct IFFHandle *iff, unsigned long type, unsigned long id)
{
    return EntryHandler(iff, type, id, IFFSLI_ROOT, &StopChunkHook, 0);
}

int
StopOnExit(struct IFFHandle *iff, unsigned long type, unsigned long id)
{
    return ExitHandler(iff, type, id, IFFSLI_ROOT, &StopChunkHook, 0);
}

/* Properties handlers */
static unsigned long
PropChunkHandler(struct Hook *hook, unsigned char *object, void *cmd)
{
    long error = 0;
    struct _IFFHandle *const _iff = (struct _IFFHandle *)object;
    struct ContextNode *cn = CurrentChunk(&_iff->iff_Pub);
    struct LocalContextItem *lci =
	AllocLocalItem(cn->cn_Type, cn->cn_ID, IFFLCI_PROP,
		       sizeof(struct StoredProperty) + cn->cn_Size);
    if (lci == NULL)
	error = IFFERR_NOMEM;
    else {
	struct StoredProperty *sp =
	    (struct StoredProperty *) LocalItemData(lci);
	sp->sp_Size = cn->cn_Size;
	sp->sp_Data = (unsigned char *)(sp + 1);
	error = ReadChunkBytes(&_iff->iff_Pub, sp->sp_Data, sp->sp_Size);
	if (error == sp->sp_Size)
	    error = StoreLocalItem(&_iff->iff_Pub, lci, IFFSLI_PROP);
	else
	    error = IFFERR_READ;
	if (error)
	    FreeLocalItem(lci);
    }
    return (unsigned long)error;
}
static struct Hook PropChunkHook = {{NULL}, 0, PropChunkHandler};

int
PropChunk(struct IFFHandle *iff, unsigned long type, unsigned long id)
{
    return EntryHandler(iff, type, id, IFFSLI_ROOT, &PropChunkHook, iff);
}

struct StoredProperty *
FindProp(struct IFFHandle *iff, unsigned long type, unsigned long id)
{
    struct LocalContextItem *lci = FindLocalItem(iff, type, id, IFFLCI_PROP);
    if (lci == NULL)
	return NULL;
    return (struct StoredProperty *)LocalItemData(lci);
}

/* Stream reading */
long
ReadChunkBytes(struct IFFHandle *iff, unsigned char *buf, long size)
{
    if (size > 0) {
	struct ContextNode *cn = CurrentChunk(iff);
	if (cn != NULL)
	    size = MIN(size, cn->cn_Size - cn->cn_Scan);
	if (size) {
	    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
	    _iff->iff_StreamAction.sc_Command = IFFCMD_READ;
	    _iff->iff_StreamAction.sc_Buf = buf;
	    _iff->iff_StreamAction.sc_NBytes = size;
	    if (CallHookPkt(_iff->iff_StreamHook, iff,
			    (void *)&_iff->iff_StreamAction))
		return IFFERR_READ;
	    /* Update current context */
	    if (cn != NULL)
		cn->cn_Scan += size;
	}
	return size;
    }
    if (size == 0)
	return 0;
    return IFFERR_READ;
}

long
ReadChunkRecords(struct IFFHandle *iff, unsigned char *buf,
		 long recsize, long numrec)
{
    struct ContextNode *cn = CurrentChunk(iff);
    long rec_remain = (cn->cn_Size - cn->cn_Scan) / recsize;
    long error;
    numrec = MIN(rec_remain, numrec);
    error = ReadChunkBytes(iff, buf, recsize * numrec);
    if (error > 0)
	error /= recsize;
    return error;
}

static int
ReadIFFLongWord(struct _IFFHandle *_iff, unsigned long *lword)
{
    unsigned char buf[4];
    long error = ReadChunkBytes((struct IFFHandle *)_iff, buf, 4);
    if (error == 4) {
	*lword = MAKE_ULONG_FROM_BUFF(buf);
	return 0;
    }
    if (error >= 0)
	error = IFFERR_READ;
    return (int)error;
}

/* Stream writting */
long
WriteChunkBytes(struct IFFHandle *iff, unsigned char *buf, long size)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    if ((_iff->iff_RWMode & IFFF_WRITE) == 0)
	return IFFERR_WRITE;
    if (size > 0) {
	struct ContextNode *cn = CurrentChunk(iff);
	if (cn != NULL && cn->cn_Size != IFFSIZE_UNKNOWN)
	    size = MIN(size, cn->cn_Size - cn->cn_Scan);
	if (size) {
	    struct _ContextNode *const _cn = (struct _ContextNode *)cn;
	    if ((_cn != NULL && _cn->cn_IsBufferred)) {
		struct ContextBuffer *cb =
		    malloc(sizeof(struct ContextBuffer) + size);
		if (cb == NULL)
		    return IFFERR_NOMEM;
		cb->cb_Size = size;
		memcpy(cb + 1, buf, size);
		AddTail((struct List *)&_cn->cn_Buffers,
			(struct Node *)&cb->cb_Node);
	    }
	    else {
		_iff->iff_StreamAction.sc_Command = IFFCMD_WRITE;
		_iff->iff_StreamAction.sc_Buf = buf;
		_iff->iff_StreamAction.sc_NBytes = size;
		if (CallHookPkt(_iff->iff_StreamHook, iff,
				(void *)&_iff->iff_StreamAction))
		    return IFFERR_WRITE;
	    }
	    /* Update current context */
	    if (cn != NULL)
		cn->cn_Scan += size;
	    return size;
	}
    }
    else if (size == 0)
	return 0;
    return IFFERR_WRITE;
}

static long
WriteIFFLongWord(struct _IFFHandle *_iff, unsigned long lword)
{
    char buf[5];
    if (WriteChunkBytes(&_iff->iff_Pub, IDtoStr(lword, buf), 4) != 4)
	return IFFERR_WRITE;
    else
	return 0;
}

/* Stream seeking */
static int
SeekIFF(struct _IFFHandle *_iff, long count)
{
    int error = 0;
    if (count != 0)
	if ((count > 0 && _iff->iff_Pub.iff_Flags & IFFF_FSEEK)
	    || _iff->iff_Pub.iff_Flags & IFFF_RSEEK) {
	    _iff->iff_StreamAction.sc_Command = IFFCMD_SEEK;
	    _iff->iff_StreamAction.sc_NBytes = count;
	    if (CallHookPkt(_iff->iff_StreamHook, _iff,
			    (void *)&_iff->iff_StreamAction) == 0) {
		struct ContextNode *cn = CurrentChunk(&_iff->iff_Pub);
		if (cn != NULL)
		    cn->cn_Scan += count;
		error = 0;
	    }
	    else
		error = IFFERR_SEEK;
	}
	else if (count > 0)
	    while (error == 0 && count != 0) {
		char buf[BUFSIZ];
		size_t n = sizeof(buf) < count ? sizeof(buf) : count;
		count -= n;
		if (ReadChunkBytes((struct IFFHandle *)_iff, buf, n) != n)
		    error = IFFERR_SEEK;
	    }
	else
	    error = IFFERR_SEEK;
    return error;
}

/* IFFHandle functions */
struct IFFHandle *
AllocIFF(void)
{
    struct _IFFHandle *_iff = malloc(sizeof(struct _IFFHandle));
    return (struct IFFHandle *)_iff;
}

void
InitIFF(struct IFFHandle *iff, long flags, struct Hook *streamHook)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    iff->iff_Flags = flags;
    _iff->iff_StreamHook = streamHook;
    NewList((struct List *)&_iff->iff_RootLCIs);
}

int
OpenIFF(struct IFFHandle *iff, long rwMode)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    iff->iff_Depth = 0;
    _iff->iff_RWMode = rwMode;
    _iff->iff_ParseState = IFFPS_INITIAL;
    NewList((struct List *)&_iff->iff_Contexts);
    return 0;
}

void
CloseIFF(struct IFFHandle *iff)
{
    while (iff->iff_Depth > 0)
	PopChunk(iff);
}

void
FreeIFF(struct IFFHandle *iff)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    struct LocalContextItem *lci;
    while ((lci = (struct LocalContextItem *)
	    RemTail((struct List *)&_iff->iff_RootLCIs)) != NULL)
	LocalItemPurge(lci);
    free(iff);
}

/* Parsing of IFF files */
static int
PushChunkRead(struct IFFHandle *iff)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    struct ContextNode *cn;
    unsigned long id, type, size;
    int error;
    /* Read the header */
    if ((error = ReadIFFLongWord(_iff, &id)) != 0)
	return error;
    if (! GoodID(id))
	return IFFERR_SYNTAX;
    if ((error = ReadIFFLongWord(_iff, &size)) != 0)
	return error;
    if (COMPOSITE_P(id)) {
	if ((cn = &PushContext(_iff, id, 0, size)->cn_Pub) == NULL)
	    return IFFERR_NOMEM;
	if ((error = ReadIFFLongWord(_iff, &type)) != 0)
	    return error;
	if (! GoodType(type))
	    return IFFERR_SYNTAX;
	cn->cn_Type = type;
	if (id == ID_PROP && FindPropContext(iff) == NULL)
	    return IFFERR_SYNTAX;
    }
    else {
	/* every non basic chunk must be in a form */
	if ((cn = CurrentChunk(iff)) == NULL || ! COMPOSITE_P(cn->cn_ID))
	    return IFFERR_SYNTAX;
	type = cn->cn_Type;
	if ((cn = &PushContext(_iff, id, type, size)->cn_Pub) == NULL)
	    return IFFERR_NOMEM;
    }
    return error;
}

static int
PopChunkRead(struct IFFHandle *iff)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    struct ContextNode *cn = CurrentChunk(iff);
    long size = cn->cn_Size;
    long scan = cn->cn_Scan;
    PopContext(_iff);
    /* Update parent context position */
    if ((cn = CurrentChunk(iff)) != NULL)
	cn->cn_Scan += scan;
    return SeekIFF(_iff, size + (size & 1) - scan);
}

/* Writting of IFF files */
static int
PushChunkWrite(struct IFFHandle *iff, unsigned long type, unsigned long id,
	       long size)
{
    const struct _ContextNode *const _pn = (struct _ContextNode *)
	CurrentChunk(iff);
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    int error = 0;

    if (_pn != NULL && ! COMPOSITE_P(_pn->cn_Pub.cn_ID))
	return IFFERR_SYNTAX;
    if (! GoodID(id) || ! GoodType(type)
	|| (size != IFFSIZE_UNKNOWN &&
	    _pn != NULL && _pn->cn_Pub.cn_Size != IFFSIZE_UNKNOWN
	    && _pn->cn_Pub.cn_Size < _pn->cn_Pub.cn_Scan + size))
	return IFFERR_SYNTAX;
    if ((_pn != NULL && _pn->cn_IsBufferred)
	|| (size == IFFSIZE_UNKNOWN && !(iff->iff_Flags & IFFF_RSEEK))) {
	struct _ContextNode *_cn = PushContext(_iff, id, type, size);
	/* If this is buffered I will write the ID and SIZE when
           popping the chunk */
	if (_cn == NULL)
	    error = IFFERR_NOMEM;
	else
	    _cn->cn_IsBufferred = 1;
    }
    else {
	if ((error = WriteIFFLongWord(_iff, id))
	    || (error = WriteIFFLongWord(_iff, size)))
	    return error;
	if (PushContext(_iff, id, type, size) == NULL)
	    return IFFERR_NOMEM;
	if (COMPOSITE_P(id))
	    error = WriteIFFLongWord(_iff, type);
    }
    return error;
}

static int
PopChunkWrite(struct IFFHandle *iff)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    struct _ContextNode *const _cn = _PopContext(_iff);
    struct ContextNode *const pn = CurrentChunk(iff);
    char pad = 0;
    long error = 0;
    long size = (_cn->cn_Pub.cn_Size == IFFSIZE_UNKNOWN
		 ? _cn->cn_Pub.cn_Scan : _cn->cn_Pub.cn_Size);
    if (_cn->cn_IsBufferred) {
	struct ContextBuffer *cb;
	if ((error = WriteIFFLongWord(_iff, _cn->cn_Pub.cn_ID)))
	    return error;
	if (COMPOSITE_P(_cn->cn_Pub.cn_ID))
	    error = WriteIFFLongWord(_iff, size + 4)
		|| WriteIFFLongWord(_iff, _cn->cn_Pub.cn_Type);
	else
	    error = WriteIFFLongWord(_iff, size);
	if (error)
	    return error;
	while ((cb = (struct ContextBuffer *)
		RemHead((struct List *)&_cn->cn_Buffers)) != NULL
	       && error == 0) {
	    error = WriteChunkBytes(iff, (unsigned char *)(cb + 1),
				    cb->cb_Size) == cb->cb_Size
		? 0 : IFFERR_WRITE;
	    free(cb);
	}
    }
    else {
	if (_cn->cn_Pub.cn_Size == IFFSIZE_UNKNOWN
	    && (error = SeekIFF(_iff, -4 - size)) == 0
	    && (error = WriteIFFLongWord(_iff, size)) == 0)
	    error = SeekIFF(_iff, size);
	/* Update parent node count */
	if (pn != NULL)
	    pn->cn_Scan += size;
    }
    if (error == 0 && size & 1)
	if ((error = WriteChunkBytes(iff, &pad, 1)) == 1)
	    error = 0;
    FreeContextNode(_cn);
    return (int)error;
}

/* Push/Pop Chunk */
int
PushChunk(struct IFFHandle *iff, unsigned long type, unsigned  long id,
	  long size)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    if (_iff->iff_RWMode & IFFF_WRITE)
	return PushChunkWrite(iff, type, id, size);
    else
	return PushChunkRead(iff);
}

int
PopChunk(struct IFFHandle *iff)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    assert(iff->iff_Depth > 0);
    if (_iff->iff_RWMode & IFFF_WRITE)
	return PopChunkWrite(iff);
    else
	return PopChunkRead(iff);
}

/* Just pushed a chunk */
static long
ParserStateEnter(struct _IFFHandle *_iff, int control, int *loop)
{
    _iff->iff_ParseState = IFFPS_ENTER;
    *loop = control == IFFPARSE_SCAN;
    if (control == IFFPARSE_SCAN || control == IFFPARSE_STEP)
	return CallEntryHandlers(_iff);
    else
	return 0;
}

/* ParseIFF: parse an IFF file.
 *
 *  If `control' is:
 *
 *  IFFPARSE_SCAN
 *   Returns control if (a) an error is encountered, (b) a stop chunk
 * is encountered or a user handler returns the special
 * `IFF_RETURN2CLIENT', (c) the end of logical file is reached,
 * in which case IFFERR_EOF is returned.
 *
 *  IFFPARSE_STEP or IFFPARSE_RAWSTEP
 *   In these two modes, ParseIFF() will return control to the caller
 * after every step in the parse, specifically, after each push of a
 * context node and just before each pop.  If returning just before a
 * pop, ParseIFF() will return IFFERR_EOC, which is not an error, per
 * se, but is just an indication that the most recent context is
 * ending.  In STEP mode, ParseIFF() will invoke the handlers for the
 * chunks, if any, before returning.  In RAWSTEP mode, ParseIFF() will
 * not invoke any handlers and will return right away.  In both cases
 * the function can be called multiple times to step through the
 * parsing of the IFF file.
 *
 */
int
ParseIFF(struct IFFHandle *iff, int control)
{
    struct _IFFHandle *const _iff = (struct _IFFHandle *)iff;
    int loop = 1;
    int error = 0;
    struct ContextNode *cn;
    if (_iff->iff_RWMode & IFFF_WRITE) {
	fputs("Error: can't ParseIFF an IFFF_WRITE\n", stderr);
	return IFFERR_READ;
    }
    assert(control == IFFPARSE_SCAN
	   || control == IFFPARSE_STEP
	   || control == IFFPARSE_RAWSTEP);
    while (loop && error == 0) {
	switch (_iff->iff_ParseState) {
	case IFFPS_INITIAL:
	    assert(iff->iff_Depth == 0);
	    if ((error = PushChunkRead(iff)) == 0) {
		/* If it doesn't start with "FORM", "LIST", or "CAT",
		   it's not an IFF-85 file. */
		cn = CurrentChunk(iff);
		assert(cn != NULL);
		if (cn->cn_ID != ID_FORM && cn->cn_ID != ID_LIST
		    && cn->cn_ID != ID_CAT) {
		    error = IFFERR_NOTIFF;
		    loop = 0;
		} else
		    error = ParserStateEnter(_iff, control, &loop);
	    }
	    break;
	case IFFPS_ENTER:
	    cn = CurrentChunk(iff);
	    assert(cn != NULL);
	    if (COMPOSITE_P(cn->cn_ID) && cn->cn_Size > cn->cn_Scan) {
		if ((error = PushChunkRead(iff)) == 0)
		    error = ParserStateEnter(_iff, control, &loop);
	    }
	    else {
		_iff->iff_ParseState = IFFPS_EXIT;
		/* Exit Handlers */
		if (control == IFFPARSE_SCAN || control == IFFPARSE_STEP) {
		    error = CallExitHandlers(_iff);
		    /* IFF_RETURN2CLIENT is only effective with
                       IFFPARSE_SCAN */
		    if (error == IFF_RETURN2CLIENT && control == IFFPARSE_STEP)
			error = 0;
		}
		else /* (control == IFFPARSE_RAWSTEP) */
		    error = IFFERR_EOC;
	    }
	    break;
	case IFFPS_EXIT:
	    _iff->iff_ParseState = IFFPS_ENTER;
	    if ((error = PopChunk(iff)) == 0)
		if (iff->iff_Depth == 0)
		    error = IFFERR_EOF;
	    break;
	}
    }
    return error == IFF_RETURN2CLIENT ? 0 : error;
}
