/*   This is RIFFL, the Replacement iffparse.library, a collection of
 *   routines to read IFF files.
 *
 *   Copyright (C) 1997 Kim-Minh Kaplan.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License version 2 as published by the Free Software Foundation.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file named COPYING for more
 *   details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __BOUNDS_CHECKING_ON
#include <unchecked.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <alib/iff-ilbm.h>
#include <proto/alib.h>
#include <proto/iffparse.h>

#include <gtk/gtk.h>
#include <libgimp/gimp.h>

static void query(void);
static void run(char *, int, GParam *, int *, GParam **);
static gint32 load_image(char *);
static gint32 load_body(struct IFFHandle *, char *);
static gint32 save_image(char *, gint32, gint32);

/* GIMP interface */
GPlugInInfo PLUG_IN_INFO =
{
    NULL,    /* init_proc */
    NULL,    /* quit_proc */
    query,   /* query_proc */
    run,     /* run_proc */
};

MAIN ();

static void
query ()
{
    static GParamDef load_args[] = {
	{ PARAM_INT32, "run_mode", "Interactive, non-interactive" },
	{ PARAM_STRING, "filename", "The name of the file to load" },
	{ PARAM_STRING, "raw_filename", "The name entered" },
    };
    static GParamDef load_return_vals[] = {
	{ PARAM_IMAGE, "image", "Output image" },
    };
    static int nload_args = sizeof (load_args) / sizeof (load_args[0]);
    static int nload_return_vals =
	sizeof (load_return_vals) / sizeof (load_return_vals[0]);

    static GParamDef save_args[] = {
	{ PARAM_INT32, "run_mode", "Interactive, non-interactive" },
	{ PARAM_IMAGE, "image", "Input image" },
	{ PARAM_DRAWABLE, "drawable", "Drawable to save" },
	{ PARAM_STRING, "filename", "The name of the file to save the image in" },
	{ PARAM_STRING, "raw_filename", "The name of the file to save the image in" },
    };
    static int nsave_args = sizeof(save_args) / sizeof(save_args[0]);

    gimp_install_procedure ("file_ilbm_load",
			    "loads files of the IFF ILBM file format",
			    "FIXME: write help",
			    "Kim-Minh Kaplan",
			    "Kim-Minh Kaplan",
			    "1997",
			    "<Load>/ILBM",
			    NULL,
			    PROC_PLUG_IN,
			    nload_args, nload_return_vals,
			    load_args, load_return_vals);
    gimp_install_procedure("file_ilbm_save",
			   "saves file in the IFF ILBM file format",
			   "FIXME: write help",
			   "Kim-Minh Kaplan",
			   "Kim-Minh Kaplan",
			   "1997",
			   "<Save>/ILBM",
			   "INDEXED*, RGB*",
			   PROC_PLUG_IN,
			   nsave_args, 0,
			   save_args, NULL);
    gimp_register_load_handler("file_ilbm_load", "ilbm,lbm", "");
    gimp_register_save_handler("file_ilbm_save", "ilbm,lbm", "");
}

static void
run (char *name, int nparams, GParam *param,
     int *nreturn_vals, GParam **return_vals)
{
    static GParam values[2];
    GRunModeType run_mode;
    gint32 image_ID;

    run_mode = param[0].data.d_int32;

    *nreturn_vals = 2;
    *return_vals = values;
    values[0].type = PARAM_STATUS;
    values[0].data.d_status = STATUS_CALLING_ERROR;
    values[1].type = PARAM_IMAGE;
    values[1].data.d_image = -1;

    if (strcmp(name, "file_ilbm_load") == 0) {
	image_ID = load_image(param[1].data.d_string);
	if (image_ID != -1) {
	    values[0].data.d_status = STATUS_SUCCESS;
	    values[1].data.d_image = image_ID;
	}
	else
	    values[0].data.d_status = STATUS_EXECUTION_ERROR;
    }
    else if (strcmp(name, "file_ilbm_save") == 0) {
	if (save_image(param[3].data.d_string, param[1].data.d_int32,
		       param[2].data.d_int32))
	    values[0].data.d_status = STATUS_SUCCESS;
	else
	    values[0].data.d_status = STATUS_EXECUTION_ERROR;
    }
}

/* IFF ILBM reading */
static gint32
load_image(char *filename)
{
    gint32 image_ID = -1;
    static const char *const progress_txt = "Loading %s: ";
    char *temp;
    struct IFFHandle *iff;
    long error;

    temp = g_malloc(strlen(progress_txt) + strlen(filename));
    sprintf(temp, progress_txt, filename);
    gimp_progress_init(temp);
    g_free(temp);

    if ((iff = AllocIFF()) != NULL) {
	InitIFFasFILE(iff);
	if ((iff->iff_Stream = (unsigned long)fopen(filename, "r")) != 0) {
	    if (OpenIFF(iff, IFFF_READ) == 0) {
		if (PropChunk(iff, ID_ILBM, ID_BMHD) == 0
		    && PropChunk(iff, ID_ILBM, ID_CMAP) == 0
		    && PropChunk(iff, ID_ILBM, ID_CAMG) == 0
		    && StopChunk(iff, ID_ILBM, ID_BODY) == 0) {
		    if ((error = ParseIFF(iff, IFFPARSE_SCAN)) == 0)
			image_ID = load_body(iff, filename);
		    else
			fprintf(stderr, "Error %ld: ParseIFF() failed.\n",
				error);
		}
		else
		    fputs("Error: PropChunk/StopChunk failed.\n", stderr);
		CloseIFF(iff);
	    }
	    else
		fputs("Error: OpenIFF() failed.\n", stderr);
	    fclose((FILE*)iff->iff_Stream);
	}
	else
	    perror(filename);
	FreeIFF(iff);
    }
    else
	fputs("Error: AllocIFF() failed.\n", stderr);
    return image_ID;
}

static unsigned short
get_word(const unsigned char *p)
{
    return (unsigned short)p[1] + 256 * (unsigned short)p[0];
}

static unsigned long
get_longword(const unsigned char *p)
{
    return (unsigned long)get_word(p + 2)
	+ 65536 * ((unsigned long)p[1] + 256 * (unsigned long)p[0]);
}

static gint32
load_body(struct IFFHandle *iff, char *filename)
{
    struct StoredProperty *sp;
    int error = 0;

    /* CMAP */
    int cmap_found = 0;
    unsigned long ncolors;
    guchar *cmap;
    
    /* CAMG */
    int camg_found = 0;
    unsigned long view_mode;

    /* BMHD */
    int bmhd_found = 0;
    unsigned int width, height, trans_col;
    int depth;
    int masking, compression;
    int nplanes;
    unsigned int bytes_per_row;

    /* BODY */
    gint32 image_ID = -1;
    gint32 layer_ID;
    GDrawable *drawable;
    GPixelRgn pixel_rgn;
    gint i, n;
    guchar *buffer, *out;
    unsigned char *linebuf = 0;

    /* CMAP */
    if ((sp = FindProp(iff, ID_ILBM, ID_CMAP)) != NULL) {
	cmap_found = 1;
	ncolors = sp->sp_Size / 3;
	cmap = sp->sp_Data;
    }

    /* CAMG */
    if ((sp = FindProp(iff, ID_ILBM, ID_CAMG)) != NULL) {
	camg_found = 1;
	view_mode = get_longword(sp->sp_Data);
    }

    /* BMHD */
    if ((sp = FindProp(iff, ID_ILBM, ID_BMHD)) == NULL) {
	fprintf(stderr, "Error: no Bitmap header\n");
	return -1;
    }
    if (sp->sp_Size < 20) {
	fprintf(stderr,
		"Error: BMHD is %ld bytes long.  I expected at least 20\n",
		sp->sp_Size);
	return -1;
    }
    bmhd_found = 1;
    width = get_word(sp->sp_Data);
    height = get_word(sp->sp_Data + 2);
    depth = sp->sp_Data[8];
    masking = sp->sp_Data[9];
    compression = sp->sp_Data[10];
    trans_col = sp->sp_Data[12];
    nplanes = depth + (masking == 1 ? 1 : 0); /* If there is a mask plane */
    bytes_per_row = 2 * ((width + 15) / 16);
    if (compression != 0 && compression != 1) {
	fprintf(stderr, "Error: Unknown body compression : %d\n", compression);
	return -1;
    }

    /* BODY */
    /* Create new image */
    image_ID = gimp_image_new(width, height, cmap_found ? INDEXED : RGB);
    gimp_image_set_filename(image_ID, filename);
    if (cmap_found)
	gimp_image_set_cmap(image_ID, cmap, ncolors);
    layer_ID = gimp_layer_new(image_ID, "Background", width, height,
			      cmap_found
			      ? (masking != 0 ? INDEXEDA_IMAGE : INDEXED_IMAGE)
			      : (masking != 0 ? RGBA_IMAGE : RGB_IMAGE),
			      100, NORMAL_MODE);
    gimp_image_add_layer(image_ID, layer_ID, 0);

    /* convert from ILBM to GIMP */
    drawable = gimp_drawable_get(layer_ID);
    gimp_pixel_rgn_init(&pixel_rgn, drawable, 0, 0,
			drawable->width, drawable->height, TRUE, FALSE);
    out = g_new(guchar, pixel_rgn.bpp * width);
    buffer = g_new(guchar, ((depth + 7) / 8) * width);
    linebuf = g_new(unsigned char, bytes_per_row * nplanes);
    for (i = 0; i < height && error == 0; i++) {
	/* Read a scan line */
	if (compression == 0)
	    error = ReadChunkBytes(iff, linebuf, bytes_per_row * nplanes)
		!= bytes_per_row * nplanes;
	else if (compression == 1)
	    for (n = 0; n < nplanes && error == 0; n++)
		error = ILBMDecodeByteRun1(iff, linebuf + n * bytes_per_row,
					   bytes_per_row);
	if (error == 0) {
	    ILBMRowToChunky(linebuf, depth, bytes_per_row,
			    buffer, width);
	    /* Scale the values */
	    for (n = 0; n < width; n++) {
		int j;
		for (j = 0; j < pixel_rgn.bpp; j++) {
		    if (cmap_found) 
			out[n * pixel_rgn.bpp + j] =
			    buffer[n * ((depth + 7) / 8) + j];
		    else {
			out[n * pixel_rgn.bpp + j] =
			    buffer[n * ((depth + 7) / 8)
				  + j * (((depth + 7) /8) / 3)];
		    }
		}
	    }
	    /* Decode mask */
	    if (masking == 1) {
		ILBMRowToChunky(linebuf + depth * bytes_per_row, 1,
					bytes_per_row, buffer, width);
		for (n = 0; n < width; n++)
		    out[(n + 1) * pixel_rgn.bpp - 1] = buffer[n] ? ~0 : 0;
	    } else if (masking == 2) {
		for (n = 0; n < width; n++)
		    out[(n + 1) * pixel_rgn.bpp - 1] =
			buffer[n] == trans_col ? 0 : ~0;
	    }
		
#ifdef __BOUNDS_CHECKING_ON
	    BOUNDS_CHECKING_OFF;
#endif
	    gimp_pixel_rgn_set_row(&pixel_rgn, out, 0, i, width);
#ifdef __BOUNDS_CHECKING_ON
	    BOUNDS_CHECKING_ON;
#endif
	    gimp_progress_update((double)i / (double)height);
	}
    }
    gimp_drawable_flush(drawable);
    g_free(linebuf);
    g_free(buffer);
    g_free(out);

    if (error) {
	gimp_image_delete(image_ID);
	return -1;
    }
    return image_ID;
}

/* IFF ILBM writing */
static void
put_word(unsigned char *p, unsigned short word)
{
    p[1] = 0xff & word;
    word /= 256;
    p[0] = 0xff & word;
}

static void
put_longword(unsigned char *p, unsigned long lword)
{
    put_word(p + 2, 0xffff & lword);
    lword /= 65536;
    put_word(p, 0xffff & lword);
}

static unsigned char
get_nbplanes(gint32 image_ID, gint32 drawable_ID)
{
    unsigned int colors;
    unsigned char nplanes = 0;
    switch (gimp_drawable_type(drawable_ID)) {
    case INDEXEDA_IMAGE: case INDEXED_IMAGE:
	for (gimp_image_get_cmap(image_ID, &colors); colors > 1; colors >>= 1)
	    nplanes++;
	break;
    case RGBA_IMAGE: case RGB_IMAGE:
	nplanes = 24;
	break;
    default:
	fprintf(stderr, "ILBM: GIMP let through an inappropriate image."
		"  Sorry, can't save this as a ILBM.\n");
	break;
    }
    return nplanes;
}

static int
has_alpha(gint32 drawable_ID)
{
    const GDrawableType t = gimp_drawable_type(drawable_ID);
    return t == INDEXEDA_IMAGE || t == GRAYA_IMAGE || t == RGBA_IMAGE;
}

static long
save_bmhd(struct IFFHandle *iff, gint32 image_ID, gint32 drawable_ID)
{
    const GDrawable *drawable = gimp_drawable_get(drawable_ID);
    unsigned char bmhd[20];
    long error;

    put_word(bmhd, drawable->width);
    put_word(bmhd + 2, drawable->height);
    put_word(bmhd + 4, 0);
    put_word(bmhd + 6, 0);
    bmhd[8] = get_nbplanes(image_ID, drawable_ID);
    bmhd[9] = has_alpha(drawable_ID) ? 1 : 0;
#ifdef ILBM_NO_COMPRESSION
    bmhd[10] = 0;		/* compression */
#else
    bmhd[10] = 1;		/* compression */
#endif
    bmhd[11] = 0;		/* pad1 */
    put_word(bmhd + 12, 0);	/* transparent color */
    bmhd[14] = 1;		/* x aspect */
    bmhd[15] = 1;		/* y aspect */
    put_word(bmhd + 16, drawable->width); /* page width */
    put_word(bmhd + 18, drawable->height); /* page height */
    error = PushChunk(iff, ID_ILBM, ID_BMHD, sizeof(bmhd));
    if (error == 0) {
	error = WriteChunkBytes(iff, bmhd, sizeof(bmhd));
	if (error == sizeof(bmhd))
	    error = 0;
    }
    if (error == 0)
	error = PopChunk(iff);
    return error;
}

static long
save_cmap(struct IFFHandle *iff, gint32 image_ID, gint32 drawable_ID)
{
    long error;
    int colors;
    guchar *cmap;
    switch (gimp_drawable_type(drawable_ID)) {
    case INDEXED_IMAGE: case INDEXEDA_IMAGE:
	cmap = gimp_image_get_cmap(image_ID, &colors);
	error = PushChunk(iff, ID_ILBM, ID_CMAP, colors * 3);
	if (error == 0) {
	    error = WriteChunkBytes(iff, cmap, colors * 3);
	    if (error == colors * 3)
		error = 0;
	}
	if (error == 0)
	    error = PopChunk(iff);
	break;
    case RGB_IMAGE: case RGBA_IMAGE:
	cmap = gimp_image_get_cmap(image_ID, &colors);
	if (cmap)
	    fputs("Uh ? Found a colormap in a RGB image\n", stderr);
	if (colors > 0)
	    fprintf(stderr, "Uh ? Found %d colors in a RGB image\n", colors);
	error = 0;
	break;
    case GRAY_IMAGE: case GRAYA_IMAGE:
	fputs("Error: ILBM can't save grayscale images\n", stderr);
	error = -1;
	break;
    default:
	fprintf(stderr, "Error: unexpected image type %d\n",
		gimp_drawable_type(drawable_ID));
	error = -1;
	break;
    }
    return error;
}

static long
save_body(struct IFFHandle *iff, gint32 image_ID, gint32 drawable_ID)
{
    GPixelRgn pixel_rgn;
    long error = 0;
    unsigned char *buffer, *out_buf;
    GDrawable *drawable = gimp_drawable_get(drawable_ID);
    const long bytes_per_row = 2 * ((drawable->width + 15) / 16);
    const int nbplanes =
	get_nbplanes(image_ID, drawable_ID) + has_alpha(drawable_ID);

    gimp_pixel_rgn_init(&pixel_rgn, drawable, 0, 0, drawable->width,
			drawable->height, FALSE, FALSE);

    if ((buffer = g_malloc(drawable->width * drawable->bpp)) == NULL)
	error = IFFERR_NOMEM;
    if (error == 0)
	if ((out_buf = g_malloc(bytes_per_row)) == NULL)
	    error = IFFERR_NOMEM;
#ifdef ILBM_NO_COMPRESSION
    if (error == 0)
	error = PushChunk(iff, ID_ILBM, ID_BODY,
			  bytes_per_row * drawable->height * nbplanes);
#else
    if (error == 0)
	error = PushChunk(iff, ID_ILBM, ID_BODY, IFFSIZE_UNKNOWN);
#endif
    if (error == 0) {
	int line;
	for (line = 0; line < drawable->height && error == 0; line++) {
	    int n;
	    unsigned long mask;
	    gimp_pixel_rgn_get_row(&pixel_rgn, buffer,
				   0, line, drawable->width);
	    /* ILBM only has on/off alpha channel */
	    for (n = 0; n < drawable->width; n++)
		if (buffer[(n + 1) * drawable->bpp - 1] & 0x80)
		    buffer[(n + 1) * drawable->bpp - 1] = ~0;
		else
		    buffer[(n + 1) * drawable->bpp - 1] = 0;
	    for (mask = 1, n = 0; n < nbplanes && error == 0; mask <<= 1, n++) {
		ILBMChunkyToRow(buffer, drawable->width, drawable->bpp, n,
				out_buf);
#ifdef ILBM_NO_COMPRESSION
		if ((error = WriteChunkBytes(iff, out_buf, bytes_per_row))
		    == bytes_per_row)
		    error = 0;
#else
		error = ILBMEncodeByteRun1(iff, out_buf, bytes_per_row);
#endif
	    }
	    gimp_progress_update((double)line / (double)drawable->height);
	}
    }
    if (error == 0)
	error = PopChunk(iff);
    return error;
}

static gint32
save_image(char *filename, gint32 image_ID, gint32 drawable_ID)
{
    static const char *const progress_txt = "Saving %s: ";
    char *temp;
    GDrawable *drawable;
    struct IFFHandle *iff;
    long error;

    temp = g_malloc(strlen(progress_txt) + strlen(filename));
    sprintf(temp, progress_txt, filename);
    gimp_progress_init(temp);
    g_free(temp);

    drawable = gimp_drawable_get(drawable_ID);

    if ((iff = AllocIFF()) != NULL) {
	InitIFFasFILE(iff);
	iff->iff_Flags |= IFFF_FSEEK | IFFF_RSEEK;
	if ((iff->iff_Stream = (unsigned long)fopen(filename, "w")) != 0) {
	    if ((error = OpenIFF(iff, IFFF_WRITE)) == 0) {
		if ((error = PushChunk(iff, ID_ILBM, ID_FORM,
				       IFFSIZE_UNKNOWN)) != 0
		    || (error = save_bmhd(iff, image_ID, drawable_ID)) != 0
		    || (error = save_cmap(iff, image_ID, drawable_ID)) != 0
		    || (error = save_body(iff, image_ID, drawable_ID)) != 0
		    || (error = PopChunk(iff)) != 0) 
		    fprintf(stderr, "Error %ld: while writing.\n", error);
		CloseIFF(iff);
	    }
	    else
		fprintf(stderr, "Error %ld: OpenIFF() failed.\n", error);
	    fclose((FILE*)iff->iff_Stream);
	}
	else {
	    perror(filename);
	    error = -1;
	}
	FreeIFF(iff);
    }
    else {
	fputs("Error: AllocIFF() failed.\n", stderr);
	error = IFFERR_NOMEM;
    }
    return error ? 0 : 1;
}
