/*   This is RIFFL, the Replacement iffparse.library, a collection of
 *   routines to read IFF files.
 *
 *   Copyright (C) 1997 Kim-Minh Kaplan.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License version 2 as published by the Free Software Foundation.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file named COPYING for more
 *   details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define USE_POSIX_STREAMS
#include <libraries/iffparse.h>
#include <utility/hooks.h>
#include <proto/iffparse.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef USE_POSIX_STREAMS
#include <fcntl.h>
#include <unistd.h>
#endif /* USE_POSIX_STREAMS */

#define ID_ILBM			MAKE_ID('I', 'L', 'B', 'M')

static long
myHandler(struct Hook *hook, unsigned char *object, long cmd)
{
    struct IFFHandle *const iff = (struct IFFHandle *)object;
    struct ContextNode *cn = CurrentChunk(iff);
    char buf[5];
    int i;
    for (i = 1; i < iff->iff_Depth; i++)
	putchar('.');
    printf("%s ", IDtoStr(cn->cn_ID, buf));
    printf("%s ", IDtoStr(cn->cn_Type, buf));
    switch (cmd) {
    case IFFCMD_ENTRY:
	printf("%ld\n", cn->cn_Size);
	break;
    case IFFCMD_EXIT:
	puts("END.");
	break;
    default:
	puts("????");
	break;
    }
    return 0;
}

static struct Hook myHook = {
    { NULL }, 0, myHandler, 0
};

int
main(int argc, char **argv)
{
    struct IFFHandle *iff = NULL;
    int opt_help = 0;
    int error = 0;
    if (argc == 2)
	opt_help = strcmp("-h", argv[1]) == 0
	    || strcmp("--help", argv[1]) == 0;
    if (argc != 2 || opt_help) {
	fprintf(opt_help ? stdout : stderr,
		"usage: %s [ -h | --help | <filename> ]\n", argv[0]);
	exit(opt_help ? EXIT_SUCCESS : EXIT_FAILURE);
    }
    iff = AllocIFF();
    if (iff == NULL) {
	fputs("Error: AllocIFF() failed.\n", stderr);
	goto bye;
    }
#ifndef USE_POSIX_STREAMS
    InitIFFasFILE(iff);
    iff->iff_Stream = (ULONG)fopen(argv[1], "r");
    if ((FILE *)iff->iff_Stream == 0) {
	perror("Error: fopen() failed");
	goto bye;
    }
#else /* USE_POSIX_STREAMS */
    InitIFFasPOSIX(iff);
    iff->iff_Stream = open(argv[1], O_RDONLY);
    if ((int)iff->iff_Stream < 0) {
	perror("Error: open() failed");
	goto bye;
    }
#endif /* USE_POSIX_STREAMS */
    error = OpenIFF(iff, IFFF_READ);
    if (error != 0) {
	fprintf(stderr, "Error %d: OpenIFF() failed.\n", error);
	goto bye;
    }
    error = EntryHandler(iff, ID_ILBM, ID_FORM, IFFSLI_ROOT, &myHook, iff);
    if (error == 0)
	error = EntryHandler(iff, ID_ILBM, ID_LIST, IFFSLI_ROOT, &myHook, iff);
    if (error == 0)
	error = EntryHandler(iff, ID_ILBM, ID_CAT, IFFSLI_ROOT, &myHook, iff);
    if (error == 0)
	error = EntryHandler(iff, ID_ILBM, ID_PROP, IFFSLI_ROOT, &myHook, iff);
    if (error) {
	fprintf(stderr, "Error %d: EntryHandler() failed.\n", error);
	goto bye;
    }
    error = ExitHandler(iff, ID_ILBM, ID_FORM, IFFSLI_ROOT, &myHook, iff);
    if (error == 0)
	error = ExitHandler(iff, ID_ILBM, ID_LIST, IFFSLI_ROOT, &myHook, iff);
    if (error == 0)
	error = ExitHandler(iff, ID_ILBM, ID_CAT, IFFSLI_ROOT, &myHook, iff);
    if (error == 0)
	error = ExitHandler(iff, ID_ILBM, ID_PROP, IFFSLI_ROOT, &myHook, iff);
    if (error) {
	fprintf(stderr, "Error %d: ExitHandler() failed.\n", error);
	goto bye;
    }
    while (1) {
	error = ParseIFF(iff, IFFPARSE_SCAN);
	if (error == IFFERR_EOC)
	    continue;		/* enter next context */
	else if (error != 0)
	    break;
    }
    if (error != 0 && error != IFFERR_EOF)
	fprintf(stderr, "Error %d: ParseIFF failed\n", error);

 bye:
    if (iff != NULL) {
	CloseIFF(iff);
#ifndef USE_POSIX_STREAMS
	if ((FILE *)iff->iff_Stream != 0)
	    fclose((FILE *)iff->iff_Stream);
#else /* USE_POSIX_STREAMS */
	if ((int)iff->iff_Stream >= 0)
	    close(iff->iff_Stream);
#endif /* USE_POSIX_STREAMS */
	FreeIFF(iff);
    }
    exit(EXIT_SUCCESS);
}
