/*   This is RIFFL, the Replacement iffparse.library, a collection of
 *   routines to read IFF files.
 *
 *   Copyright (C) 1997 Kim-Minh Kaplan.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License version 2 as published by the Free Software Foundation.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file named COPYING for more
 *   details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define USE_POSIX_STREAMS
#include <libraries/iffparse.h>
#include <utility/hooks.h>
#include <proto/iffparse.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef USE_POSIX_STREAMS
#include <fcntl.h>
#include <unistd.h>
#endif /* USE_POSIX_STREAMS */

static void
PrintTopChunk(struct IFFHandle *iff, FILE *fd)
{
    const struct ContextNode *top = CurrentChunk(iff);
    char buf[5];
    int i;
    if (top == NULL)
	return;
    for (i = 1; i < iff->iff_Depth; i++)
	putc('.', fd);
    fprintf(fd, "%s %ld ", IDtoStr(top->cn_ID, buf), top->cn_Size);
    if (top->cn_ID == ID_FORM || top->cn_ID == ID_LIST
	|| top->cn_ID == ID_CAT || top->cn_ID == ID_PROP)
	fprintf(fd, "%s", IDtoStr(top->cn_Type, buf));
    putc('\n', fd);
}

int
main(int argc, char **argv)
{
    struct IFFHandle *iff = NULL;
    int opt_help = 0;
    int error = 0;
    if (argc == 2)
	opt_help = strcmp("-h", argv[1]) == 0
	    || strcmp("--help", argv[1]) == 0;
    if (argc != 2 || opt_help) {
	fprintf(opt_help ? stdout : stderr,
		"usage: %s [ -h | --help | <filename> ]\n", argv[0]);
	exit(opt_help ? EXIT_SUCCESS : EXIT_FAILURE);
    }
    iff = AllocIFF();
    if (iff == NULL) {
	fputs("Error: AllocIFF() failed.\n", stderr);
	goto bye;
    }
#ifndef USE_POSIX_STREAMS
    InitIFFasFILE(iff);
    iff->iff_Stream = (ULONG)fopen(argv[1], "r");
    if ((FILE *)iff->iff_Stream == 0) {
	perror("Error: fopen() failed");
	goto bye;
    }
#else /* USE_POSIX_STREAMS */
    InitIFFasPOSIX(iff);
    iff->iff_Stream = open(argv[1], O_RDONLY);
    if ((int)iff->iff_Stream < 0) {
	perror("Error: open() failed");
	goto bye;
    }
#endif /* USE_POSIX_STREAMS */
    error = OpenIFF(iff, IFFF_READ);
    if (error != 0) {
	fputs("Error: OpenIFF() failed.\n", stderr);
	goto bye;
    }

    while (1) {
	error = ParseIFF(iff, IFFPARSE_RAWSTEP);
	if (error == IFFERR_EOC)
	    continue;		/* enter next context */
	else if (error != 0)
	    break;
	PrintTopChunk(iff, stdout);
    }
    if (error != IFFERR_EOF)
	fprintf(stderr, "Error %d: ParseIFF failed\n", error);

 bye:
    if (iff != NULL) {
	CloseIFF(iff);
#ifndef USE_POSIX_STREAMS
	if ((FILE *)iff->iff_Stream != 0)
	    fclose((FILE *)iff->iff_Stream);
#else /* USE_POSIX_STREAMS */
	if ((int)iff->iff_Stream >= 0)
	    close(iff->iff_Stream);
#endif /* USE_POSIX_STREAMS */
	FreeIFF(iff);
    }
    exit(EXIT_SUCCESS);
}
